function [gx,hx,retcode] = FRWZ_iterativeprocedure(GLOBALS,~,~,~,~,P,A,B)
% FRWZ_iterativeprocedure.m
% 
% performs the Foerster, Rubio-Ramirez, Waggoner, and Zha (2013)
%   iterative procedure for solving first-order peturbation of MS models
% 
% edited 2014/04/07
% Andrew T. Foerster
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %
% 
%  INPUTS
%   nvec = 6-vector with elements [ny nx ne nt1 nt2 ns]:
%       ny = dim(y) the number of nonpredetermined variables
%       nx = dim(x) the number of predetermined variables
%       ne = dim(e) the number of exogenous shocks
%       nt1 = dim(theta1) the number of switching parameters affecting the
%           steady state
%       nt2 = dim(theta2) the number of switching parameters not affecting
%           the steady state
%       ns = the number of regimes in the model
%   P = (ns x ns) matrix where P(i,j)=Pr(s_{t+1}=j|s_{t}=i)
%   Dmatrix = (ns x ns) cell with elements Dmatrix{s',s}, where
%       Dmatrix{s',s} is a (nx+ny x 2*(nx+ny+ne+nt))-matrix and
%       is the Jacobian of the equilibrium conditions w.r.t.
%       [y_{t+1},y_{t},x_{t},x_{t-1},eps_{t+1},eps_{t},theta_{t+1)(s'),theta_{t}(s)]
%       evaluated at steady state
%  guess = 
%  
%  OUTPUTS
%   gx = (ns x 1) cell with elements gx{i}, gx{i} is a (ny x nx)-matrix
%       gx{i} is the response of y_{t} to x_{t-1} conditional on regime i
%   hx = (ns x 1) cell with elements hx{i}, hx{i} is a (nx x nx)-matrix
%       hx{i} is the response of x_{t} to x_{t-1} conditional on regime i
%   retcode  = boolean taking values
%       0: solution not found, iterative procedure does not converge
%       1: solution found, iterative procedure converges  
% 
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %



% -- Setup -- %
ny = GLOBALS.ny;
nx = GLOBALS.nx;
ns = GLOBALS.ns;
crit = 1e-10;
maxit = 200;


% -- Constructing Matrices -- %
gx = cell(ns,1);
hx = cell(ns,1);

% guess
for s = 1:ns
    gx{s} = zeros(ny,nx);
    hx{s} = zeros(nx,nx);
end

jj = 0;
norm = Inf;
while norm > crit && jj < maxit
    jj = jj + 1;
    gx0 = gx;
    hx0 = hx;   
    
    % now iteratively solve
    s = 1;
    stop = 0;
    while s <= ns && stop == 0
        Amat = zeros(nx+ny,nx+ny);
        Bmat = zeros(nx+ny,nx+ny);
        for sp = 1:ns        
            if sp == s
                Amat = Amat + [P(s,sp)*A{sp,s}(:,1:nx) P(s,sp)*A{sp,s}(:,nx+1:nx+ny)];
            else
                Amat(:,1:nx) = Amat(:,1:nx) + P(s,sp)*A{sp,s}(:,1:nx) + P(s,sp)*A{sp,s}(:,nx+1:nx+ny)*gx{sp};
            end
            Bmat = Bmat - P(s,sp)*B{sp,s};
        end
        
        % eigenvalue decomposition
        if isinf(max(max(abs([Amat Bmat])))) || isnan(max(max(abs([Amat Bmat]))))
            gx{s} = Inf*ones(ny,nx);
            hx{s} = Inf*ones(nx,nx);
            retcode = 0;
            warning('Infinite Matrix')
            return;
        else
            [V, D] = eig(Bmat,Amat);
        end

        %reorder eigenvalues to be in ascending order
        [~,I] = sort(abs(diag(D)));
        d = diag(D);
        D = diag(d(I));
        V = V(:,I);
        % if max(max(abs(Amat*V*D - Bmat*V))) > 1e-8 
        % 	warning(['Eigenvalue Decomposition Accuracy Error: ' num2str(max(max(abs(Amat*V*D - Bmat*V))))]); 
        % end
        
        % solve system if a solution exists (ie invertibility is met)
        if rank(V(1:nx,1:nx)) == nx
            gx{s} = V(nx+1:nx+ny,1:nx)/(V(1:nx,1:nx));
            hx{s} = V(1:nx,1:nx)*D(1:nx,1:nx)/(V(1:nx,1:nx));
        else
            warning('Rank Condition Not Met - No Solution');
            gx{s} = Inf*ones(ny,nx);
            hx{s} = Inf*ones(nx,nx);
            stop = 1;
        end
        if max(max(abs(imag([gx{s};hx{s}])))) < 1e-8       
            gx{s} = real(gx{s});
            hx{s} = real(hx{s});
        else
            warning('Large Imaginary Portions of Decision Rules');
            gx{s} = Inf*ones(ny,nx);
            hx{s} = Inf*ones(nx,nx);
            stop = 1;
        end
        
        % increment s
        s = s + 1;
    end 
    
    % check norm if solved correctly, otherwise terminate loop
    if stop == 0
        norm = 0;
        for s = 1:ns
            norm = max([norm;max(max(abs([gx{s}-gx0{s};hx{s}-hx0{s}])))]);
        end
    else
        norm = 0;
        jj = maxit;
    end
end

if jj == maxit
    retcode = 0;
    if GLOBALS.FRWZprinttime == 1
        warning(['Iterative procedure did not converge, norm = ' num2str(norm)]);
    end
else
    retcode = 1;
    if GLOBALS.FRWZprinttime == 1
        disp(['Iterative procedure converged after ' num2str(jj) ' iterations, norm = ' num2str(norm)]);
    end
end


    